function r=renyi(tfr,t,f,alpha);
%RENYI	Measure Renyi information.
%	R=RENYI(TFR,T,F,ALPHA) measures the Renyi information relative 
%	to a Type 1 time-frequency energy distribution (negative and
%       positive frequencies)
%
%	TFR : time-frequency energy distribution.
%	T : time vector (default : (1:N)).	
%	F : frequency vector (default : (-M/2:(M/2-1))/M).
%	ALPHA : rank of the Renyi measure (default : 3).
%	R : the alpha-rank Renyi measure (in bits if TFR is a time- 
%	    frequency matrix) :
%	     R=log2[Sum[TFR(Fi,Ti)^ALPHA dFi.dTi]/(1-ALPHA)]
%                  Fi,Ti
%
%	Example :
%	 s=atoms(64,[32,.3,16,1]); [TFR,T,F]=tfrsp(s); R=renyi2(TFR,T,F,3) 
%
%	 s=atoms(64,[16,.2,10,1;40,.4,12,1]); [TFR,T,F]=tfrsp(s); 
%	 R=renyi2(TFR,T,F,3) 

%	P. Goncalves, October 95
%	Copyright (c) 1995 Rice University.
%       E. Chassande, May 1998
%
%	------------------- CONFIDENTIAL PROGRAM -------------------- 
%	This program can not be used without the authorization of its
%	author(s). For any comment or bug report, please send e-mail to 
%	f.auger@ieee.org 

if (nargin == 0),
 error('At least one parameter required');
end;
[M,N] = size(tfr);
tfr=[tfr(M/2+1:M,:);tfr(1:M/2,:)];

if (nargin == 1),
 t=1:N; f=(-M/2:(M/2-1))'/M; alpha=3;
elseif (nargin == 2),
 f=(-M/2:(M/2-1))'/M; alpha=3;
elseif (nargin == 3),
  f=[f(M/2+1:M);f(1:M/2)]; alpha=3;
else
  f=[f(M/2+1:M);f(1:M/2)];
end;

dt=t(2:N)-t(1:N-1); 
Mini=min(dt); Maxi=max(dt);
if (Mini~=Maxi),
 error('The time instants must be regularly sampled.');
else
 dt=Mini;
end;

df=f(2:M)-f(1:M-1); 
Mini=min(df); Maxi=max(df);
if (Mini~=Maxi),
 error('The frequency bins must be regularly sampled.');
else
 df=Mini;
end;

d=dt*df;
% [f indic]=sort(f);
% tfr=tfr(indic,:);
E=d*sum(sum(tfr));
tfr=tfr/E;

if alpha == 1,
 if (min(min(tfr))<0),
  error('ALPHA=1, TFR must be strictly positive');
 else
  r=-d*sum(sum(tfr.*log2(tfr+eps)));
 end
else
 r=log2(d*sum(sum(tfr.^alpha))+eps)/(1-alpha) ;
end







