function [tout,t,x,xref,f,nout,d,slope] = plchirp(m1,m2,r,tin,fout,fs);
% PLCHIRP.M
%
% P. Flandrin, 05/12/97. Last modification 05/19/97.
%
% This function evaluates the waveform of a gravitational
% wave expected to be radiated by a coalescing binary
% in a Newtonian approximation (coalescence time = 0).
%
% Input parameters:
% m1   = mass of object 1, in solar masses
% m2   = mass of object 2, in solar masses
% r    = distance binary-earth, in Mpc
% tin  = time to coalescence, in seconds
% fout = high cut-off frequency, in Hertz (< fs/2))
% fs   = sampling frequency, in Hertz
%
% Output parameters:
% tout = time at which fout is reached, in seconds
% t    = time vector, in seconds
% x    = waveform
% xref = reference waveform (matched filter)
% f    = instantaneous frequency of x, in Hertz
% nout = index at which fout is reached
%
% Typical example: Innocent/Torresani 1,
%
% m1=10; m2=10; r=1000; 
% fmin=50; fmax=500;
% tin=.515; fs=1000;
% [tout,t,x,sref,f,nout,d] = plchirp(m1,m2,r,tin,fmax,fs);
% a=1.2*max(abs(x));
% plot(t,x); axis([t(1) tout -a a]);
% xlabel('time (seconds)'); ylabel('amplitude');

% total mass
M = m1 +m2;
alpha=1/4;
beta=5/8;

% reduced mass
mu = 1/(1/m1 + 1/m2)

% chirp mass
calM = M^(2/5)*mu^(3/5)

% tout
tout = 3*calM^(-5/3)*(fout/100)^(-8/3)

% d
d = 160*3^(3/8)*calM^(-5/8)

% amplitude
A = (400/3^(3/4))*1.92*10^(-23)*calM^(5/4)/r;
%A=abs(beta-1)^(1/2)*(d*beta)^((-alpha+1/2)/(beta-1))

% waveform
ts = 1/fs;
t = (-tin:ts:0);
tcol=length(t);
x=zeros(tcol,1);
t2= (-tin:ts:-tout)';
nout=length(t2);
x(1:nout) = A*(abs(t2)).^(-alpha).*cos(2*pi*d*(abs(t2)).^(beta));
C=A*abs(beta-1)^(-1/2)*(d*beta)^((alpha-1/2)/(beta-1))

% instantaneous frequency
f  = zeros(tcol,1);
f(1:nout) = (5*d/8)*(abs(t2)).^(-3/8);

% reference with C normalized (C=1)
A=abs(beta-1)^(1/2)*(d*beta)^((-alpha+1/2)/(beta-1));
xref=zeros(nout,1);
xref=A*(abs(t2)).^(-alpha).*cos(2*pi*d*(abs(t2)).^(beta));

slope=(alpha-1+beta/2)/(beta-1);
