function [tfrsp, tfrrsp, tfrsrsp, S, K] = tfrsrsp(x,t,f,N,lh,indiclh,w,pw,trace);
%TFRSRSP selective reassignment of spectrograms.
%	 [TFRSP TFRRSP TFRSRSP]=TFRsrsp(X,T,N,H,LH,INDICLH,W,PW,TRACE) 
%        computes a modified version of a combination of spectrograms
%        using a selective reassignment principle. The analysis
%        windows are the columns of the matrix H.
% 
%        X       : signal
%	 LH      : frequency smoothing windows lengths
%	 INDICLH : indices of windows used for the combination
%		   of spectrograms and reassigment maps
%	 W       : name of the smoothing window
%        PW      : smoothing window parameters
%	 T       : time instant(s).
%	 F       : frequency locations.
%	 N       : number of frequency bins
%	 TRACE   : if nonzero, the progression of the algorithm is shown.
%	 TFR     : time-frequency representation. When called without 
%                  output arguments, TFRSRSP runs TFRQVIEW.
%	 	   TFRSP is the arithmetic mean of the set of
%	  	   spectrograms associated with lh.
%	 S       : supervision matrix
%	 K	 : Kullback-Leibler distance matrix
%
%	Example :
% 	 s=fmlin(64,0,0.5);
%	 noi=randn(64,1);sig=sigmerge(s,noi,10);
%	 t=32-20:32+20;f=1:32;N=64;
%        [tfr1, tfr2, tfr3, d] = tfrsrsp(sig,t,f,N,9:2:65,13:17,'gauss',1e-3,1);
%	 subplot(2,2,1);imagesc(t,f/N,displog(tfr1,-30));axis('xy'); colormap(jet);
%	 xlabel('time');ylabel('frequency');title('spectrogram');
%	 subplot(2,2,2);imagesc(t,f/N,displog(tfr2,-30));axis('xy');
%	 xlabel('time');ylabel('frequency');title('reassigned spectrogram');
%	 subplot(2,2,3);imagesc(t,f/N,displog(tfr3,-30));axis('xy');
%	 xlabel('time');ylabel('frequency');title('supervised reassignment');
%	 subplot(2,2,4);imagesc(t,f/N,d);axis('xy');
%	 xlabel('time');ylabel('frequency');title('decision map');
%
%	See also all the time-frequency representations listed in
%	 the file CONTENTS (TFR*)
%

%	 Another example :
%        s=fmlin(128,0,0.5);
%	 noi=randn(128,1);sig=sigmerge(s,noi,10);
%	 t=64-32:64+32;f=1:64;N=128;
%        [tfr1, tfr2, tfr3, d] = tfrsrsp(sig,t,f,N,9:4:125,13:17,'gauss',1e-3,1);
%
%	E. Chassande-Mottin, F. Auger, February 1996.

[xrow,xcol] = size(x);
[trow,tcol] = size(t);
[frow,fcol] = size(f);
[hrow hcol] = size(lh);

if (nargin < 4),
 error('At least 4 parameters are required');
elseif (nargin==4),
 trace=0;
end;

if (xcol~=1),
 error('X must have only one column');
elseif (trow~=1),
 error('T must only have one row');
elseif (N<=0),
 error('N must be greater than zero'); 
elseif (2^nextpow2(N)~=N),
 fprintf('For a faster computation, N should be a power of two\n');
elseif (frow~=1),
  error('F must only have one row');
elseif (hrow~=1),
  error('LH must only have one row');
elseif sum(indiclh>length(lh)),
  error('Every elements of INDICLH must be lower than the length of lh'); 
end; 
clear xcol trow hrow frow hrow

% calcul de l'energie du signal et du seuil de reallocation
Ex=mean(abs(x(t)).^2); Threshold=1.0e-6*Ex;

if (sum(rem(lh,2))~=hcol),
 error('LH must be a vector with odd numbers only');
end;

if (tcol==1),
 Dt=1; 
else
 Deltat=t(2:tcol)-t(1:tcol-1); 
 Mini=min(Deltat); Maxi=max(Deltat);
 if (Mini~=Maxi),
  error('The time instants must be regularly sampled.');
 else
  Dt=Mini;
 end;
 clear Deltat Mini Maxi;
end;

tfrsp   = zeros (fcol,tcol);% initialisation des matrices de rtf
tfrrsp  = zeros (fcol,tcol);
tfrsrsp = zeros (fcol,tcol);
tf1     = zeros (fcol,tcol);
tf2     = zeros (fcol*tcol,hcol);
tf3     = zeros (N,tcol);
th      = zeros (hcol,1);
fh      = zeros (hcol,1);
sp      = zeros (N,tcol);
rsp     = zeros (N,tcol);

if trace, disp('supervised reassigned spectrogram'); end;
if trace, disp('spectrograms'); end;
% spectrogramme et operateurs de reallocation

indiclh2=1:hcol;
indiclh2(indiclh)=NaN*ones(size(indiclh));
indiclh2=indiclh2(finite(indiclh2));

for ih=indiclh2,
 if trace, disprog(ih,hcol,10); end
 h=window(lh(ih),w,pw);
 h=h/norm(h);
 [tm th(ih)]=loctime(h);th(ih)=th(ih)/(2*sqrt(pi));
 [fm fh(ih)]=locfreq(h);fh(ih)=fh(ih)/(2*sqrt(pi));
 tf3=opreassp(x,t,N,h,0);
 tf3=tf3(f,:); tf2(:,ih)=tf3(:);
end;

for ih=indiclh,
 h=window(lh(ih),w,pw); 
 h=h/norm(h);
 [tm th(ih)]=loctime(h);th(ih)=th(ih)/(2*sqrt(pi));
 [fm fh(ih)]=locfreq(h);fh(ih)=fh(ih)/(2*sqrt(pi));
 [sp rsp tf3]=tfrrsp2(x,t,N,h,0);
 tfrsp=tfrsp+sp(f,:); clear sp; 
 tfrrsp=tfrrsp+rsp(f,:); clear rsp;
 tf3=tf3(f,:);
 tf1=tf1+tf3; 
 tf2(:,ih)=tf3(:);
 clear tf3;
end;

tfrrsp=tfrrsp/length(indiclh);
tfrsp=tfrsp/length(indiclh);
tf1=tf1/length(indiclh);

% frequence reduite/adimensionne
tf2=real(tf2)*diag(1./th)+i*imag(tf2)*diag(1./fh)/N;
% critere de decision de reallocation
K=zeros(1,tcol*fcol);
phi=zeros(tcol*fcol,1);
N1=20;

if trace, disp('supervision'); end;
for icol=1:tcol*fcol, 
 if trace, disprog(icol,tcol*fcol,10); end
 if tfrsp(icol)>Threshold,
  phi=angle(tf2(icol,:));
  % mise en forme de l'angle
  phi=abs(phi);
  indice=find((phi>pi/2)&(phi<=pi));
  phi(indice)=pi-phi(indice);
  % signal si : l'angle dans [0 pi/2] est distribue de cette maniere...
  phi1=phi((phi>0)&(phi<pi/2));
  if length(phi1)>2,  
   [dphi1 phi1x]=hist(phi1,N1);dphi1=dphi1/(sum(dphi1)/(phi1x(2)-phi1x(1)));
   dphiref1=1/(max(lh)-min(lh))/2*(1+tan(phi1x).^2)./abs(tan(phi1x)).^(3/2);
   dphiref1=dphiref1/(sum(dphiref1)/(phi1x(2)-phi1x(1)));
   % distance de Kullback
   indice=find(dphi1~=0); % 0.log(0)=0
   K(icol)=sum(dphi1(indice).*log(dphi1(indice)./dphiref1(indice)))/(phi1x(2)-phi1x(1));
  else K(icol)=0; % si tous les points sont en 0 ou en pi/2...
  end;
 else K(icol)=nan; % rien a reallouer...
 end;
end;

K=reshape(K,fcol,tcol);

% decision 1: reallouer, 0: laisse sur place
kmax=0.8; % K<kmax => reallouer
% 0.8 pour super2

tf=tf1;                     %
% ok=1                         %
% while ok==1,
% keyboard;                    %

tfrsrsp = zeros (fcol,tcol); %
S=(K<=kmax);
tf1=tf1.*S;                  

% tf1=tf.*S;             

if trace, disp('reassigment'); end;
% reallocation
for icol=1:tcol
if trace, disprog(icol,tcol,10); end
 for jcol=1:fcol
  icolhat=icol+round(real(tf1(jcol,icol)));
  if icolhat<1, icolhat=1;end;  
  if icolhat>tcol, icolhat=tcol;end;
  jcolhat=jcol-round(imag(tf1(jcol,icol)));
  if jcolhat<1, jcolhat=1;end;  
  if jcolhat>fcol, jcolhat=fcol;end;
  tfrsrsp(jcolhat,icolhat)=tfrsrsp(jcolhat,icolhat)+tfrsp(jcol,icol);
 end;
end;

% clf;
% imagesc(displog(tfrsrsp,-40));
% axis('xy');
% colormap(flipud(gray));
% end;                        %

if (nargout==0),
 continue=1;
 while (continue==1),
  choice=menu ('Choose the representation:',...
               'stop',...
               'spectrogram',...
               'reassigned spectrogram',...
               'selectively reassigned spectrogram');
  if (choice==1), continue=0;
  elseif (choice==2), 
   tfrqview(tfrsp,t,'tfrsp',h(:,1));
  elseif (choice==3),
   tfrqview(tfrrsp,t,'tfrrsp',h(:,1));
  elseif (choice==4),
   tfrqview(tfrsrsp,t,'tfrrsp',h(:,1));
  end;
 end;
end;





