function [tfr, rtfr, a, f, tf2]=tfrrscalo3(x,t,a_desc,wavelet,param,trace)
%TFRRSCALO Reassigned scalogram.
%	[TFR,RTFR,A,F,TF2]=TFRRSCALO(X,T,A_DESC,WAVELET,PARAM,TRACE) computes 
%	the scalogram (squared magnitude of a continuous wavelet
%	transform) and its reassigned version.
%
%	X : signal, 
%	T : time instant(s) (default : 1:length(X)),
%	A_DESC : scale vector descriptor (default : [2*F0 N*F0 64 0]),
%       A_DESC = [AMIN AMAX N DYADIC],
%       AMIN and AMAX define the analysis scale span,
%       N is the number of scale (or frequency) bins,
%	A dyadic (resp. linear) scale axis is used if 
%       DYAD is set to 1 (resp. 0),
%	WAVELET : name of the wavelet (default : 'klauder'),
%	Possible names are :
%	'Klauder', 'Morlet','MexHat'
%	PARAM : optional parameter
%	For the Klauder wavelet, [LAMBDA,K] where
%	 LAMBDA : attenuation factor or the envelope (default : 50)
%	 FO     : central frequency of the mother wavelet (default : 1/length(X))
%	 K      : sets the value at both extremities (default : 1e-3)
%	For the Morlet wavelet, [FOT,K]
%	 FOT    : time-bandwidth product of the mother wavelet (default : 2.1)
%	 K      : sets the value at both extremities (default : 1e-3)
%	For the Mexican hat wavelet, no additional parameter.
%	TRACE : if nonzero, the progression of the algorithm is shown
%	                                 	(default : 0),
%	TFR : time-frequency decomposition.
%	A   : scale vector
%	F   : frequency vector
%
%	Example :    
%	 sig=fmlin(64,0.1,0.5); 
%	 [tfr rtfr f tf2]=tfrrscalo3(sig);
%        pcolor(1:64,f,rtfr);axis('xy');
%
%	See also all the time-frequency representations listed in
%	the file CONTENTS (TFR*)

%	E. Chassande-Mottin, November 1996. 
%	Copyright (c) 1996.
%
%	------------------- CONFIDENTIAL PROGRAM -------------------- 
%	This program can not be used without the authorization of its
%	author(s). For any comment or bug report, please send e-mail to 
%			    lemoine@alto.unice.fr 
if (nargin == 0),
 error('At least one parameter required');
end;

[xrow,xcol] = size(x);
if nargin == 1,
 t=1:xrow; wavelet='klauder'; param=[]; trace=0;
elseif nargin == 2,
 wavelet='klauder'; param=[]; trace=0;
elseif nargin == 3, 
 wavelet='klauder'; param=[]; trace=0;
elseif nargin == 4,
 param=[]; trace=0;
elseif nargin == 5,
 trace=0;
end;
trace=1;
[trow,tcol] = size(t);

if (xcol~=1),
 error('X must have only one column');
elseif (trow~=1),
 error('T must only have one row');
end;

wavelet=upper(wavelet);
if strcmp(wavelet,'KLAUDER') | strcmp(wavelet,'KLAU'),
 if length(param)==0,
  lambda0=50; f0=1/xrow; K=1e-3; %f0=0.2; K=1e-3;
 elseif length(param)==1,
  lambda0=param(1); f0=1/xrow; K=1e-3;
 elseif length(param)==2,
  lambda0=param(1); f0=param(2); K=1e-3;
 elseif length(param)==3,
  lambda0=param(1); f0=param(2); K=param(3);
 else
  error('KLAUDER wavelet need only three optional parameters');
 end;
 beta = 2*pi*lambda0*f0-1/2;
 csth = gamma(beta+1)*2^(beta+0.5)/sqrt(2*pi*gamma(2*beta+1));
 cstDh= gamma(beta+2)*2^(beta+0.5)/sqrt(2*pi*gamma(2*beta+1));
elseif strcmp(wavelet,'MORLET') | strcmp(wavelet,'MORL'),
elseif strcmp(wavelet,'MEXHAT') | strcmp(wavelet,'MEXH'),
else error('unknown wavelet');
end;

if nargin > 3,
 [arow acol]=size(a_desc);
 if (acol~=4)&(arow~=1),
  error('A_DESC must be a 4 element row vector');
 else
  amin = a_desc(1);   amax = a_desc(2);
  N    = 2*a_desc(3); dyad = a_desc(4);
  if amin<2*f0,
   error('AMIN must be greater than 2*F0');
  elseif amax<amin,
   error('AMAX must be greater than AMIN');
  end;
 end;
else
 amin = 2*f0;     amax = xrow*f0;
 N    = xrow;     dyad = 0;
end;

if (tcol==1),
 Dt=1; 
else
 Deltat=t(2:tcol)-t(1:tcol-1); 
 Mini=min(Deltat); Maxi=max(Deltat);
 if (Mini~=Maxi),
  error('The time instants must be regularly sampled.');
 else
  Dt=Mini;
 end;
 clear Deltat Mini Maxi;
end;

clear xcol a_desc arow acol trow

if trace, disp('Scalogram'); end;
if dyad,
 lgmin= log(amin)/log(2); lgmax= log(amax)/log(2);
 a=2.^(lgmin:(lgmax-lgmin)/(N/2-1):lgmax);
 plot(a);
 if trace,
  disp(['log scale axis with ' num2str(lgmax-lgmin)...
  ' octaves and ' num2str(N/2/(lgmax-lgmin)) ' voices/octave']);
 end;
else
 if trace, disp('linear scale axis'); end;
 a=linspace(amin,amax,N/2);
end
a=[a -fliplr(a)];

if trace,
 if strcmp(wavelet,'KLAUDER') | strcmp(wavelet,'KLAU'),
  M=ceil(sqrt((csth*(lambda0*a(N/2))^(beta+0.5)/K)^(2/(beta+1))...
    -(lambda0*a(N/2))^2));
 end;
 echmax=round(min(min(t)/M,(xrow-max(t))/M));
 disp(['max scale without border effect: ',num2str(echmax)]);
 disp(['frequency span: [Fmin= ' num2str(f0/amax)...
 ', Fmax= ' num2str(f0/amin) ']']);
end;

tfr   = zeros(N,tcol);
tf2   = zeros(N,tcol);
tf3   = zeros(N,tcol);

for n=1:N/2
 if trace, disprog(n,N/2,10), end;
 if strcmp(wavelet,'KLAUDER') | strcmp(wavelet,'KLAU'),
  lambda=lambda0*a(n); 
  normh=csth*lambda^(beta+0.5);
  M=ceil(sqrt((normh/K)^(2/(beta+1))-lambda^2));
  tau=-M:M;  hstar = normh*(lambda+i*tau).^(-beta-1);
  Thstar= tau.*hstar;                         % /a(n) *a(n)
  normDh=cstDh*lambda^(beta+0.5);
  Dhstar= normDh*i*(lambda+i*tau).^(-beta-2); % *a(n) /a(n)
 end;
 for icol=1:tcol,
   ti= t(icol); tau =-min([M,ti-1]):min([M,xrow-ti]);
   tfr(n,icol)=  hstar(M+1+tau)*x(ti+tau);
   tf2(n,icol)= Thstar(M+1+tau)*x(ti+tau);
   tf3(n,icol)= Dhstar(M+1+tau)*x(ti+tau);
   tfr(N+1-n,icol)= conj( hstar(M+1+tau))*x(ti+tau);
   tf2(N+1-n,icol)= conj(Thstar(M+1+tau))*x(ti+tau);
   tf3(N+1-n,icol)= conj(Dhstar(M+1+tau))*x(ti+tau);
 end;
end;

avoid_warn=find(tfr~=0.0);
tf2(avoid_warn)=tf2(avoid_warn)./tfr(avoid_warn);
tf3(avoid_warn)=tf3(avoid_warn)./tfr(avoid_warn);
tfr=abs(tfr).^2;

if trace, disp('reassignment:'); end;
rtfr= zeros(N,tcol); 
Ex=mean(abs(x(min(t):max(t))).^2); Threshold=1.0e-6*Ex;
f= f0./a; 
if dyad,
 factor=(N/2-1)/(log(amax)-log(amin));
else
 factor=(N/2-1)/(amax-amin);
end;
for icol=1:tcol,
 if trace, disprog(icol,tcol,10); end;
 for jcol=1:N,
  if tfr(jcol,icol)>Threshold,
   icolhat= icol + real(tf2(jcol,icol)/Dt);
   icolhat= min(max(icolhat,1),tcol);
   jcolhat= imag(tf3(jcol,icol))/2/pi;
   jcolhat= rem(rem(jcolhat,1)+1,1);
   if jcolhat<0.5,
    % positive frequencies/scales
    if dyad,
     jcolhat= 1+(log(f0/jcolhat)-log(amin))*factor;
    else
     jcolhat= 1+(f0/ jcolhat   -amin)*factor;
    end;
    jcolhat= min(max(jcolhat,1),N/2);
   else    
    % negative frequencies/scales
    if dyad,
     jcolhat= N-(log(-f0/(jcolhat-1))-log(amin))*factor;
    else
     jcolhat= N+(f0/(jcolhat-1)+amin)*factor;
    end;
    jcolhat= min(max(jcolhat,N/2),N);
   end;
   tf2(jcol,icol)= jcolhat + j * icolhat;
   jcolhat=round(jcolhat);
   icolhat=round(icolhat);
   rtfr(jcolhat,icolhat)= rtfr(jcolhat,icolhat)+tfr(jcol,icol);
  else
   tf2(jcol,icol)= (1+j)*inf;
   tf3(jcol,icol)= (1+j)*inf;
   rtfr(jcol,icol)=rtfr(jcol,icol)+tfr(jcol,icol);
  end;
 end;
end;
