function [part, attrac, tfr, tf2]= tfrpart(x,t,N,h,k,trace);
%TFRPART Partition of the time-frequency plane.
%	[PART, ATTRAC, TFR, TF2]= TFRPART(X,T,N,H,K,METHOD,TRACE);
%	computes a partition of the time-frequency
%	plane within attraction areas of minima of STFT
%	log-modulus.
% 
%	X     : analysed signal.
%	T     : the time instant(s)      (default: 1:length(X)).
%	N     : number of frequency bins (default: length(X)).
%	H     : frequency smoothing window, H(0) being forced to 1
%	                                 (default: Hamming(N/4)).
%	K     : iteration step for Runge-Kutta algorithm.
%	TRACE : if nonzero, the progression of the algorithm is shown
%	                                 (default: 0).
%	PART  : time-frequency partition,
%	ATTRAC: center of each partition class,
%	TFR   : time-frequency distribution,
%	TF2   : 
%
%	Example 1:
%	s=atoms(64,[20,0.3,21,1;25,0.1,12,1;45,0.2,5,1]);
%	s=fmlin(64,0,0.5);
%	h=atoms(21,[10,0,sqrt(64),1]);
%	[part center tfr tf2]=tfrpart(s,1:64,64,h,0.4,1);
%	imagesc(part);axis('xy');colormap(flipud(gray));
%	hold on;plot(center,'y+');hold off;
%
%	Example 2:
%	sig=fmlin(100,0,0.5);
%	sig=atoms(100,[50,0.4,20,1;45,0.1,10,1]);
%       h=atoms(19,[10,0,sqrt(32),1]);
%	[part center tfr tf2]=tfrpart(sig,50+(-16:15),32,h,0.4,1);
%	imagesc(part);axis('xy');colormap(flipud(gray));
%	hold on;plot(center,'y+');hold off;
%	See also  all the time-frequency representations listed in
%	the file CONTENTS (TFR*)
%
%	E. Chassande-Mottin, F. Auger, Version 1.0, 28 oct. 1996.
%
%	------------------- CONFIDENTIAL PROGRAM -------------------- 
%	This program can not be used without the authorization of its
%	author(s). For any comment or bug report, please send e-mail to 
%			     

if (nargin == 0),
 error('At least 1 parameter required');
end;
[xrow,xcol] = size(x);

hlength=floor(N/4);
hlength=hlength+1-rem(hlength,2);

if (nargin == 1),
 t=1:xrow; N=2^nextpow2(xrow); h = window(hlength); k=0.4; trace=0;
elseif (nargin == 2),
  N=2^nextpow2(xrow); h = window(hlength); k=0.4; trace=0;
elseif (nargin == 3),
 h = window(hlength); k=0.4; trace = 0;
elseif (nargin == 4),
 k=0.4; trace = 0;
elseif (nargin == 5),
 trace = 0;
end;

if (N<0),
 error('N must be greater than zero');
end;

[trow,tcol] = size(t);
if (xcol~=1),
 error('X must have only one column');
elseif (trow~=1),
 error('T must only have one row'); 
elseif (2^nextpow2(N)~=N),
 fprintf('For a faster computation, N should be a power of two\n');
end; 

[hrow,hcol]=size(h); Lh=(hrow-1)/2; 
if (hcol~=1)|(rem(hrow,2)==0),
 error('H must be a smoothing window with odd length');
end;

if (tcol==1),
 Dt=1; 
else
 Deltat=t(2:tcol)-t(1:tcol-1); 
 Mini=min(Deltat); Maxi=max(Deltat);
 if (Mini~=Maxi),
  error('The time instants must be regularly sampled.');
 else
  Dt=Mini;
 end;
 clear Deltat Mini Maxi;
end;

tfr= zeros(N,tcol); tf2= zeros(N,tcol); tf3= zeros (N,tcol);
if trace, disp('Spectrogram'); end;
Th=h.*[-Lh:Lh]'; Dh=dwindow(h);
for icol=1:tcol,
 ti= t(icol); 
 tau=-min([N/2-1,Lh,ti-1]):min([N/2-1,Lh,xrow-ti]);
 indices= rem(N+tau,N)+1;
 if trace, disprog(icol,tcol,10); end;
 norm_h=norm(h(Lh+1+tau));
 tfr(indices,icol)=x(ti+tau).*conj( h(Lh+1+tau)) /norm_h;
 tf2(indices,icol)=x(ti+tau).*conj(Th(Lh+1+tau)) /norm_h;
 tf3(indices,icol)=x(ti+tau).*conj(Dh(Lh+1+tau)) /norm_h;
end ;
tfr=fft(tfr); tf2=fft(tf2); tf3=fft(tf3);
avoid_warn=find(tfr~=0);
tf2(avoid_warn)=real(tf2(avoid_warn)./tfr(avoid_warn)/Dt);
tf3(avoid_warn)=-imag(N*tf3(avoid_warn)./tfr(avoid_warn)/(2.0*pi));
tfr=abs(tfr).^2;

Ex=mean(abs(x(min(t):max(t))).^2); Threshold=1.0e-3*Ex;
avoid_warn=find(tfr>Threshold);
hat=zeros(N,tcol);
hat(avoid_warn)=tf2(avoid_warn)+i*tf3(avoid_warn);
clear tf3 avoid_warn tau indices;

if trace, fprintf ('\nreassignment: \n'); end;
Threshdist=1e-2; kitermax=300;

for icol=1:tcol,
 if trace, disprog(icol,tcol,10); end;
 for jcol=1:N,
  xn=icol+i*jcol; 
  % history=xn;
  if hat(jcol,icol),
   kiter=1; deltan=0; continue=1;
   while continue, 
    while(imag(xn)>N), xn=real(xn)+1j*(imag(xn)-N); end;
    while(imag(xn)<1), xn=real(xn)+1j*(imag(xn)+N); end;

    titer=real(xn); fiter=imag(xn);
    t1iter=min(max(floor(titer),1),tcol-1); t2iter=t1iter+1;
    f1iter=min(max(floor(fiter),1),N-1   ); f2iter=f1iter+1;
    % fprintf('%8.3f %8.3f, %8.3f %8.3f,',t1iter, t2iter, f1iter, f2iter);

    H11=hat(f1iter,t1iter); 
    H12=hat(f1iter,t2iter);
    H21=hat(f2iter,t1iter);
    H22=hat(f2iter,t2iter);

    CommonDenominator=(t2iter-t1iter)*(f2iter-f1iter);
    alpha11=(t2iter-titer)*(f2iter-fiter);
    alpha12=(titer-t1iter)*(f2iter-fiter);
    alpha21=(t2iter-titer)*(fiter-f1iter);
    alpha22=(titer-t1iter)*(fiter-f1iter);

    % calcul de Hxn=hat(xn)
    Hxn=(alpha11*H11+alpha12*H12+alpha21*H21+alpha22*H22)/CommonDenominator;
    xnbar=xn+k*Hxn; 

    xnbar=min(max(real(xnbar),1),tcol)+1j*imag(xnbar);
    while(imag(xnbar)>N), xnbar=real(xnbar)+1j*(imag(xnbar)-N); end;
    while(imag(xnbar)<1), xnbar=real(xnbar)+1j*(imag(xnbar)+N); end;

    tbar=real(xnbar); fbar=imag(xnbar);
    t1bar=min(max(floor(tbar),1),tcol-1); t2bar=t1bar+1;
    f1bar=min(max(floor(fbar),1),N-1   ); f2bar=f1bar+1;

    % fprintf('%8.3f %8.3f, %8.3f %8.3f\n',t1bar, t2bar, f1bar, f2bar);

    H11=hat(f1bar,t1bar); 
    H12=hat(f1bar,t2bar);
    H21=hat(f2bar,t1bar);
    H22=hat(f2bar,t2bar);

    CommonDenominator=(t2bar-t1bar)*(f2bar-f1bar);
    alpha11=(t2bar-tbar)*(f2bar-fbar);
    alpha12=(tbar-t1bar)*(f2bar-fbar);
    alpha21=(t2bar-tbar)*(fbar-f1bar);
    alpha22=(tbar-t1bar)*(fbar-f1bar);

    % calcul de Hxnbar=hat(xnbar)
    Hxnbar=(alpha11*H11+alpha12*H12+alpha21*H21+alpha22*H22)/CommonDenominator;

    xnm1=xn;
    xn=xn+0.5*k*(Hxn+Hxnbar);
    xn=min(max(real(xn),1),tcol)+1j*imag(xn);
    while(imag(xn)>N), xn=real(xn)+1j*(imag(xn)-N); end;
    while(imag(xn)<1), xn=real(xn)+1j*(imag(xn)+N); end;

    % stop ou encore ?
    deltanm1= deltan;
    deltan  = xn-xnm1;
    kiter=kiter+1;
    continue=((real(deltan)^2 > abs(real(deltanm1-deltan))*Threshdist) | ...
             (imag(deltan)^2 > abs(imag(deltanm1-deltan))*Threshdist)) & ...
             (kiter<kitermax);
    if (kiter==kitermax), fprintf('kitermax reached\n'); end;
    % history=[history xn];
   end;
  end;
  % fprintf('%8.3f %8.3f, ',jcol, icol);
  % fprintf('%8.3f %8.3f, %8.3f\n',real(xn), imag(xn), kiter);
  % subplot(211); plot(real(history));
  % subplot(212); plot(imag(history)); pause;
  tf2(jcol,icol)=xn;
 end;
end;

indic=tfr>Threshold; 
Data=tf2(indic); Data=Data(:).';
DataLength=length(Data)

fprintf('\nPre-processing... \n');
square= zeros(size(Data));

% Initialization
nsqu=1;
center= Data(1);
ridge = 4*Threshdist;
square(1) =1;

% Loop
for icol=2:DataLength,
 if trace, disprog(icol,DataLength,10), end;
 flag=0; isqu=1;
 while (flag==0)&(isqu<=nsqu),
  Data2Cent=Data(icol)-center(isqu);
  if (abs(real(Data2Cent))<=ridge(isqu)/2)&(abs(imag(Data2Cent))<=ridge(isqu)/2),
   %fprintf('\ninsert point... \n');
   square(icol)=isqu;
   pisq=Data(square==isqu);
   pxmax=max(real(pisq)); pxmin=min(real(pisq));
   pymax=max(imag(pisq)); pymin=min(imag(pisq));
   center(isqu)=(pxmin+pxmax)/2+1j*(pymin+pymax)/2;
   ridgex=pxmin+2*Threshdist-(pxmax-2*Threshdist);
   ridgey=pymin+2*Threshdist-(pymax-2*Threshdist);
   ridge(isqu) =min(ridgex,ridgey);
   flag=1;
   %pause
  end;
  isqu=isqu+1;
 end;
 if flag==0,
  %fprintf('\nnew square... \n');
  nsqu=nsqu+1;
  square(icol)=nsqu;
  center=[center Data(icol)];
  ridge =[ridge 4*Threshdist];
  %pause
 end;
end;

%clf;
%plot(Data,'+');
%hold on;
%plot(center,'o');
%for isqu=1:nsqu,
% span=[-ridge(isqu)/2 ridge(isqu)/2]; 
% top=[ridge(isqu)/2 ridge(isqu)/2];
% plot(real(center(isqu))+span,imag(center(isqu))+top);
% plot(real(center(isqu))+span,imag(center(isqu))-top);
% plot(real(center(isqu))+top,imag(center(isqu))+span);
% plot(real(center(isqu))-top,imag(center(isqu))+span);
%end;
%hold off;
%axe=max(abs(axis));
%axis([-axe axe -axe axe]); axis('square');

clear flag pxmax pxmin pymax pymin ridgex ridgey
clear Data Data2Cent isqu  pisq 

centerLength=length(center)

fprintf('\nHierarchical ascending classification \n');

CahTree=zeros(centerLength,1);
ColTree=(1:centerLength).';
ColSet =ColTree;

% compute Distmatrix
ii=0;
for i=1:centerLength-1,
 DistMatrix(ii+(1:centerLength-i))=...
   abs(center(i+1:centerLength)-center(i)*ones(1,centerLength-i));
 ii=ii+centerLength-i;
end;

MaxDist=max(max(DistMatrix));
Remainingcenter=centerLength;

% first iteration : two values together

% Look for the two nearest points  
[Minimum, MinI]=min(DistMatrix);
imin=1; 
while (MinI>=centerLength), 
 imin=imin+1; 
 MinI=MinI-centerLength+imin; 
end; 
jmin=MinI+1;

%clustering is finished when all elements are in the same cluster
while (any(CahTree==0|CahTree~=CahTree(1)))&(Minimum<(1+Threshdist)),
 if trace, disprog(sum(ColSet==0)+1,centerLength,10), end;

 if (CahTree(imin)==0),
  if (CahTree(jmin)==0),
   % regroupement de 2 singletons
   CahTree(imin)=Minimum;
   CahTree(jmin)=Minimum;
   ColTree(imin)=ColTree(jmin);
   ColSet(imin) =0;

   Indice=(jmin-1)+0.5*(imin-1)*(2*centerLength-2-imin);
   DistMatrix(Indice)=10.0*MaxDist;
   Remainingcenter=Remainingcenter-2;
  else
   % i est un singleton, j une classe
   indicesj=find(CahTree==CahTree(jmin));
   % keyboard;
   CahTree(imin)=Minimum;
   CahTree(indicesj)=Minimum*ones(size(indicesj));
   ColTree(imin)=ColTree(indicesj(1));
   ColSet(imin) =0;
   
   for jj=1:length(indicesj),
    indi=imin;
    indj=indicesj(jj);
    if (indi>indj),
     temp=indi; indi=indj; indj=temp;
    end;
    Indice=(indj-1)+0.5*(indi-1)*(2*centerLength-2-indi);
    DistMatrix(Indice)=10.0*MaxDist;
   end;
   Remainingcenter=Remainingcenter-1;
  end;
 else
  indicesi=find(CahTree==CahTree(imin));
  if (CahTree(jmin)==0),
   % i est une classe, j un singleton
   CahTree(indicesi)=Minimum*ones(size(indicesi));
   CahTree(jmin)=Minimum;
   ColTree(indicesi)=ColTree(jmin)*ones(size(indicesi));
   ColSet(indicesi)=zeros(size(indicesi));
   
   for ii=1:length(indicesi),
    indi=indicesi(ii);
    indj=jmin;
    if (indi>indj),
     temp=indi; indi=indj; indj=temp;
    end;
    Indice=(indj-1)+0.5*(indi-1)*(2*centerLength-2-indi);
    DistMatrix(Indice)=10.0*MaxDist;
   end;
   Remainingcenter=Remainingcenter-1;
  else
   % i et j sont deux classes
   indicesj=find(CahTree==CahTree(jmin));

   CahTree(indicesi)=Minimum*ones(size(indicesi));
   CahTree(indicesj)=Minimum*ones(size(indicesj));
   ColTree(indicesi)=ColTree(indicesj(1))*ones(size(indicesi));
   ColSet(indicesi) =zeros(size(indicesi));
   
   for ii=1:length(indicesi),
    for jj=1:length(indicesj),
     indi=indicesi(ii);
     indj=indicesj(jj);
     if (indi>indj),
      temp=indi; indi=indj; indj=temp;
     end;
     Indice=(indj-1)+0.5*(indi-1)*(2*centerLength-2-indi);
     if (Indice==0), 
      fprintf('Il y a encore un probleme, prevenir F. Auger=0\n');
      keyboard; 
     end;
     DistMatrix(Indice)=10.0*MaxDist;
    end;
   end;
  end;
 end;
 % Look for the two nearest points
 [Minimum, MinI]=min(DistMatrix);
 imin=1;
 while (MinI>=centerLength),
  imin=imin+1;
  MinI=MinI-centerLength+imin;
 end;
 jmin=MinI+1;
 if (imin==jmin), pause; end;
end;

%keyboard;
ColSet=ColSet(ColSet>0);
npart=length(ColSet);

attrac=zeros(npart,1);
part=zeros(N,tcol);
partvec=zeros(DataLength,1);

for icol=1:npart,
 clsqu=find(ColTree==ColSet(icol));
 nsqu =length(clsqu);
 attrac(icol)=sum(center(clsqu))/nsqu;
 for isqu=1:nsqu,
  pisq=find(square==clsqu(isqu));
  partvec(pisq)=icol*ones(size(pisq));
 end;
end;

part(indic)=partvec;
