function [t,x,f,slope,crit1,crit2] = plcfig(alpha,beta,d,A,tin,fout,fs,dr);
% PLCFIG evaluates the waveform of a chirp using the
% following definition:
%    x(t) = A |t|^(-alpha) cos(2 pi d |t|.^(beta))
%
% Input parameters:
% alpha= enveloppe exponent
% beta = instantaneaous frequency exponent
% d    = modulation rate
% A    = amplitude
% tin  = time to coalescence, in seconds
% fout = high cut-off frequency, in Hertz (< fs/2))
% fs   = sampling frequency, in Hertz
% dr   = dynamical range (optional)
%        dr=0 : no display
%        dr~=0 display (signal, inst. freq., stat. phase approx)
%        dr is used for the last plot.
% Output parameters:
% tout = time at which fout is reached, in seconds
% t    = time vector, in seconds
% x    = waveform
% f    = instantaneous frequency of x, in Hertz
% slope= theoretical slope for the spectrum in a log-log plot
%        predicted by the phase stat. approximation
% crit1= constant in Harthong criterion for phase stat. approximation
% crit2= constant in heuristic criterion for phase stat. approximation
%        the complete criterion is
%          critere{1/2}=crit{1/2} * f^(-beta/(beta-1)) * (beta d)^(1/(beta-1))
%
% Example :
% [t,x,f,slope] = plcfig(0.15,0.85,100,1,4,250,2500,40);
%

%  P. Flandrin, 05/12/97. Last modification 05/19/97.
%  E. Chassande-Mottin, 5/3/98
%

d = abs(d)*sign(beta);

ts = 1/fs;

tout=(fout/abs(d*beta))^(1/(beta-1));
Npoints = 2^(nextpow2(floor((tin-tout)/ts)));
t = fliplr(-tout-ts*(0:Npoints-1));
tcol=length(t);
x=zeros(tcol,1);
x(1:tcol) = A*(abs(t)).^(-alpha).*exp(-i*2*pi*d*(abs(t)).^(beta));

% instantaneous frequency
f  = zeros(tcol,1);
f(1:tcol)  = beta*d*(abs(t)).^( beta-1);

slope=-(alpha-1+beta/2)/(beta-1);

% display
if dr~=0,
  xnorm = x; 
  T = length(xnorm);
  N = nextpow2(T); 
  fxnorm = fft(xnorm,2^N);
  freq = linspace(0,fs/2,2^(N-1)+1);
  spec = abs(fxnorm(1:2^(N-1)+1)).^2;
  spec = spec/max(max(spec));
  l    = length(freq);
  % keyboard
  
  % plot of the stationary phase approximation of the spectrum
  subplot(2,1,1)
  fmin = min(f(f>freq(2)));fmax = min(max(f),fs/2);
  fminax=10^(floor(log10(fmin))); % frequence min. pour les axes
  indfmin = min(find(freq>=fmin));indfmax = min(find(freq>=fmax));
  fc = min(find(freq > sqrt(fmin*fmax)));
  specd = (log10(spec(fc)))*10-(log10(fmax)-log10(fmin))*10*slope;
  specf = (log10(spec(fc)))*10+(log10(fmax)-log10(fmin))*10*slope;
  semilogx([fmin fmin],[-dr specd],'r');
  hold on
  semilogx([fmax fmax],[-dr specf],'r');
  semilogx([fmin fmax],[specd specf],'r')
  
  xlabel('frequency (Hz)')
  ylabel('DSP (dB)')

  % plot of the spectrum
  semilogx(freq(2:l),10*log10(spec(2:l)),'--');
  semilogx(freq(indfmin:indfmax),10*log10(spec(indfmin:indfmax)));
  axis([fminax fs/2 -dr 0]);
  hold off

  % plot of criteria
  subplot(2,1,2);
  crit1=(5/48/pi)*(12*alpha.^2-12*alpha+12*alpha*beta+2*beta^2-5*beta+2)/(beta-1);
  d0=d*beta;
  critere1 = abs(crit1*freq.^(-beta/(beta-1))*d0^(1/(beta-1)));
  crit2 = max(alpha,abs(beta-1))/(2*pi);
  critere2 = abs(crit2*freq.^(-beta/(beta-1))*d0^(1/(beta-1)));
  semilogx(freq(indfmin:indfmax),critere1(indfmin:indfmax));
  grid on
  hold on
  semilogx(freq(indfmin:indfmax),critere2(indfmin:indfmax),'--');
  axis([fminax fs/2 0 min(1,max(max(critere1),max(critere2)))]);
  hold off
  ylabel('critere')
  xlabel('frequency (Hz)')
end;